/*
 * Copyright (C) 2011 Igalia S.L.
 * Copyright (C) 2008 Luca Bruno <lethalman88@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#if !defined(__WEBKIT_H_INSIDE__) && !defined(BUILDING_WEBKIT)
#error "Only <wpe/webkit.h> can be included directly."
#endif

#ifndef WebKitError_h
#define WebKitError_h

#include <wpe/WebKitDefines.h>

G_BEGIN_DECLS

#define WEBKIT_NETWORK_ERROR                     webkit_network_error_quark ()
#define WEBKIT_POLICY_ERROR                      webkit_policy_error_quark ()
#define WEBKIT_DOWNLOAD_ERROR                    webkit_download_error_quark ()
#define WEBKIT_JAVASCRIPT_ERROR                  webkit_javascript_error_quark ()
#define WEBKIT_SNAPSHOT_ERROR                    webkit_snapshot_error_quark ()
#define WEBKIT_USER_CONTENT_FILTER_ERROR         webkit_user_content_filter_error_quark ()
#define WEBKIT_WEB_EXTENSION_ERROR               webkit_web_extension_error_quark ()
#define WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR webkit_web_extension_match_pattern_error_quark ()


#define WEBKIT_MEDIA_ERROR                       webkit_media_error_quark ()

/**
 * WebKitNetworkError:
 * @WEBKIT_NETWORK_ERROR_FAILED: Generic load failure
 * @WEBKIT_NETWORK_ERROR_TRANSPORT: Load failure due to transport error
 * @WEBKIT_NETWORK_ERROR_UNKNOWN_PROTOCOL: Load failure due to unknown protocol
 * @WEBKIT_NETWORK_ERROR_CANCELLED: Load failure due to cancellation
 * @WEBKIT_NETWORK_ERROR_FILE_DOES_NOT_EXIST: Load failure due to missing file
 *
 * Enum values used to denote the various network errors.
 **/
typedef enum {
    WEBKIT_NETWORK_ERROR_FAILED = 399,
    WEBKIT_NETWORK_ERROR_TRANSPORT = 300,
    WEBKIT_NETWORK_ERROR_UNKNOWN_PROTOCOL = 301,
    WEBKIT_NETWORK_ERROR_CANCELLED = 302,
    WEBKIT_NETWORK_ERROR_FILE_DOES_NOT_EXIST = 303
} WebKitNetworkError;

/**
 * WebKitPolicyError:
 * @WEBKIT_POLICY_ERROR_FAILED: Generic load failure due to policy error
 * @WEBKIT_POLICY_ERROR_CANNOT_SHOW_MIME_TYPE: Load failure due to unsupported mime type
 * @WEBKIT_POLICY_ERROR_CANNOT_SHOW_URI: Load failure due to URI that can not be shown
 * @WEBKIT_POLICY_ERROR_FRAME_LOAD_INTERRUPTED_BY_POLICY_CHANGE: Load failure due to frame load interruption by policy change
 * @WEBKIT_POLICY_ERROR_CANNOT_USE_RESTRICTED_PORT: Load failure due to port restriction
 *
 * Enum values used to denote the various policy errors.
 **/
typedef enum {
    WEBKIT_POLICY_ERROR_FAILED = 199,
    WEBKIT_POLICY_ERROR_CANNOT_SHOW_MIME_TYPE = 100,
    WEBKIT_POLICY_ERROR_CANNOT_SHOW_URI = 101,
    WEBKIT_POLICY_ERROR_FRAME_LOAD_INTERRUPTED_BY_POLICY_CHANGE = 102,
    WEBKIT_POLICY_ERROR_CANNOT_USE_RESTRICTED_PORT = 103
} WebKitPolicyError;


/**
 * WebKitDownloadError:
 * @WEBKIT_DOWNLOAD_ERROR_NETWORK: Download failure due to network error
 * @WEBKIT_DOWNLOAD_ERROR_CANCELLED_BY_USER: Download was cancelled by user
 * @WEBKIT_DOWNLOAD_ERROR_DESTINATION: Download failure due to destination error
 *
 * Enum values used to denote the various download errors.
 */
typedef enum {
    WEBKIT_DOWNLOAD_ERROR_NETWORK = 499,
    WEBKIT_DOWNLOAD_ERROR_CANCELLED_BY_USER = 400,
    WEBKIT_DOWNLOAD_ERROR_DESTINATION = 401
} WebKitDownloadError;


/**
 * WebKitJavascriptError:
 * @WEBKIT_JAVASCRIPT_ERROR_SCRIPT_FAILED: An exception was raised in JavaScript execution
 * @WEBKIT_JAVASCRIPT_ERROR_INVALID_PARAMETER: An unsupported parameter has been used to call and async function from API. Since 2.40
 * @WEBKIT_JAVASCRIPT_ERROR_INVALID_RESULT: The result of JavaScript execution could not be returned. Since 2.40
 *
 * Enum values used to denote errors happening when executing JavaScript
 */
typedef enum {
    WEBKIT_JAVASCRIPT_ERROR_SCRIPT_FAILED = 699,
    WEBKIT_JAVASCRIPT_ERROR_INVALID_PARAMETER = 600,
    WEBKIT_JAVASCRIPT_ERROR_INVALID_RESULT = 601,
} WebKitJavascriptError;

/**
 * WebKitSnapshotError:
 * @WEBKIT_SNAPSHOT_ERROR_FAILED_TO_CREATE: An error occurred when creating a webpage snapshot.
 *
 * Enum values used to denote errors happening when creating snapshots of #WebKitWebView
 */
typedef enum {
    WEBKIT_SNAPSHOT_ERROR_FAILED_TO_CREATE = 799
} WebKitSnapshotError;

/**
 * WebKitWebExtensionError:
 * @WEBKIT_WEB_EXTENSION_ERROR_UNKNOWN: An unknown error occured.
 * @WEBKIT_WEB_EXTENSION_ERROR_RESOURCE_NOT_FOUND: A specified resource was not found on disk.
 * @WEBKIT_WEB_EXTENSION_ERROR_INVALID_RESOURCE_CODE_SIGNATURE: A resource failed the bundle's code signature checks.
 * @WEBKIT_WEB_EXTENSION_ERROR_INVALID_MANIFEST: An invalid `manifest.json` was encountered.
 * @WEBKIT_WEB_EXTENSION_ERROR_UNSUPPORTED_MANIFEST_VERSION: The manifest version is not supported.
 * @WEBKIT_WEB_EXTENSION_ERROR_INVALID_MANIFEST_ENTRY: An invalid manifest entry was encountered.
 * @WEBKIT_WEB_EXTENSION_ERROR_INVALID_DECLARATIVE_NET_REQUEST_ENTRY: An invalid declarative net request entry was encountered.
 * @WEBKIT_WEB_EXTENSION_ERROR_INVALID_BACKGROUND_PERSISTENCE: The extension specified background persistence that was not compatible with the platform or features requested.
 * @WEBKIT_WEB_EXTENSION_ERROR_INVALID_ARCHIVE: The archive file is invalid or corrupt.
 *
 * Enum values used to denote errors happening when parsing a #WebKitWebExtension
 * 
 * Since: 2.52
 */
typedef enum {
    WEBKIT_WEB_EXTENSION_ERROR_UNKNOWN = 899,
    WEBKIT_WEB_EXTENSION_ERROR_RESOURCE_NOT_FOUND = 800,
    WEBKIT_WEB_EXTENSION_ERROR_INVALID_RESOURCE_CODE_SIGNATURE = 801,
    WEBKIT_WEB_EXTENSION_ERROR_INVALID_MANIFEST = 802,
    WEBKIT_WEB_EXTENSION_ERROR_UNSUPPORTED_MANIFEST_VERSION = 803,
    WEBKIT_WEB_EXTENSION_ERROR_INVALID_MANIFEST_ENTRY = 804,
    WEBKIT_WEB_EXTENSION_ERROR_INVALID_DECLARATIVE_NET_REQUEST_ENTRY = 805,
    WEBKIT_WEB_EXTENSION_ERROR_INVALID_BACKGROUND_PERSISTENCE = 806,
    WEBKIT_WEB_EXTENSION_ERROR_INVALID_ARCHIVE = 807,
} WebKitWebExtensionError;

/**
 * WebKitWebExtensionMatchPatternError:
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_UNKNOWN: An unknown error occured.
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_INVALID_SCHEME: The scheme component was invalid.
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_INVALID_HOST: The host component was invalid.
 * @WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_INVALID_PATH: The path component was invalid.
 *
 * Enum values used to denote errors happening when creating a #WebKitWebExtensionMatchPattern
 * 
 * Since: 2.52
 */
typedef enum {
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_UNKNOWN = 899,
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_INVALID_SCHEME = 808,
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_INVALID_HOST = 809,
    WEBKIT_WEB_EXTENSION_MATCH_PATTERN_ERROR_INVALID_PATH = 810,
} WebKitWebExtensionMatchPatternError;

/**
 * WebKitUserContentFilterError:
 * @WEBKIT_USER_CONTENT_FILTER_ERROR_INVALID_SOURCE: The JSON source for a content filter is invalid.
 * @WEBKIT_USER_CONTENT_FILTER_ERROR_NOT_FOUND: The requested content filter could not be found.
 *
 * Errors that can occur while compiling content filters.
 *
 * Since: 2.24
 */
typedef enum {
    WEBKIT_USER_CONTENT_FILTER_ERROR_INVALID_SOURCE,
    WEBKIT_USER_CONTENT_FILTER_ERROR_NOT_FOUND,
} WebKitUserContentFilterError;

/**
 * WebKitMediaError:
 * @WEBKIT_MEDIA_ERROR_WILL_HANDLE_LOAD: Preliminary load failure for media content types. A new load will be started to perform the media load. Since: 2.40
 *
 * Enum values used to denote the various media errors.
 *
 * Since: 2.40
 */
typedef enum {
    WEBKIT_MEDIA_ERROR_WILL_HANDLE_LOAD = 204
} WebKitMediaError;

WEBKIT_API GQuark
webkit_network_error_quark                     (void);

WEBKIT_API GQuark
webkit_policy_error_quark                      (void);


WEBKIT_API GQuark
webkit_download_error_quark                    (void);


WEBKIT_API GQuark
webkit_javascript_error_quark                  (void);

WEBKIT_API GQuark
webkit_snapshot_error_quark                    (void);

WEBKIT_API GQuark
webkit_web_extension_error_quark               (void);

WEBKIT_API GQuark
webkit_web_extension_match_pattern_error_quark (void);

WEBKIT_API GQuark
webkit_user_content_filter_error_quark         (void);

WEBKIT_API GQuark
webkit_media_error_quark                       (void);

G_END_DECLS

#endif
